/*
 RemoveUnnecessaryMaskKeyframes.jsx
 version: 1.0
 author: Charles Bordenave (www.nabscripts.com)
 date: 7 Jan 2020
*/


function RemoveUnnecessaryMaskKeyframes()
{
	var removeUnnecessaryMaskKeyframes = this;

	var utils = new RemoveUnnecessaryMaskKeyframesUtils();

	// infos
	this.scriptName = "Remove Unnecessary Mask Keyframes";
	this.scriptVersion = "1.0";
	this.scriptCopyright = "Copyright (c) 2021 Charles Bordenave";
	this.scriptHomepage = "http://www.nabscripts.com";
	this.scriptDescription = "This script allows you to remove \\'unnecessary\\' keyframes generated by the built-in mask tracker.\\r\\rTwo passes are executed: removal and reinforcement. The removal pass consists in removing keyframes that are too similar to the previous one. Two keyframes are considered similar when none of the vertices has moved more than the specified (removal) threshold.\\rThe reinforcement pass analyzes every successive pair of the remaining keyframes. When a vertex has moved more than the (reinforcement) threshold, a previously removed keyframe is reintroduced in the middle of the two keyframes. This phase is repeated until every pair satisfies the condition.\\r\\rThe percentage of keyframe reduction is written to the Info palette.\\r\\r";
	this.scriptAbout = this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + this.scriptDescription;
	this.scriptUsage = "\u25BA Specify the removal threshold value in pixels.\\r" +
					   "\u25BA Specify the reinforcement threshold value in pixels.\\r" +
					   "\u25BA Click on Apply \\r";

	// errors and messages
	this.selErr = "Please select a mask first.";
	this.verticesErr = "The number of mask vertices must be the same for every keyframe.";
	this.processErr = "An error occurred during the process.";
	this.infoMsg1 = "Removal: %d";
	this.infoMsg2 = "Reinforcement: %d";
	this.infoMsg3 = "Reduction amount: %d%";
	
	// help tips
	this.removalThresholdHlp = "Higher value removes more keyframes";
	this.reinforcementThresholdHlp = "Lower value adds more keyframes";
	this.removalThresholdInfoStr = "Removal Threshold:\nThis threshold sets the minimum distance traveled by at least one vertex from one keyframe to another. If none of the vertices has traveled this distance, then the second keyframe is removed.\nHigher value means removing more keyframes.";
	this.reinforcementThresholdInfoStr = "Reinforcement Threshold:\nThis threshold applies after the Removal Threshold. To avoid drift between remaining keyframes and better stick to the original mask, previously removed keyframes are reintroduced between every pair of keyframes for which a vertex has traveled more than the given threshold.\nLower value means adding more keyframes.";
	
	// default settings
	this.removalThreshold = 3; // distance in pixels
	this.reinforcementThreshold = 6;
	this.removalThresholdPrefSetting = "Removal Threshold";
	this.reinforcementThresholdPrefSetting = "Reinforcement Threshold";
	
	// look for existing prefs
	if (app.settings.haveSetting(this.scriptName, this.removalThresholdPrefSetting))
	{
		this.removalThreshold = parseFloat(app.settings.getSetting(this.scriptName, this.removalThresholdPrefSetting));
	}
	if (app.settings.haveSetting(this.scriptName, this.reinforcementThresholdPrefSetting))
	{
		this.reinforcementThreshold = parseFloat(app.settings.getSetting(this.scriptName, this.reinforcementThresholdPrefSetting));
	}
	
	this.buildUI = function (thisObj)
	{
		// dockable panel or palette
		var pal = (thisObj instanceof Panel) ? thisObj : new Window("palette", this.scriptName, undefined, {resizeable:true});

		// resource specifications
		var res =
		"group { spacing: 5, orientation:'column', alignment:['fill','top'], alignChildren:['right','top'], \
			gr1: Group { spacing: 5, alignment:['right','top'], alignChildren:['right','center'], \
				removalThresholdSt: StaticText { text:'Removal Threshold:' }, \
				gr11: Group { alignment:['left','top'], alignChildren:['right','center'], \
					removalThresholdEt: EditText { text:'" + this.removalThreshold + "', characters:5, alignment:['fill','center'], helpTip:'" + this.removalThresholdHlp + "' }, \
					helpBtn: Button { text:'?', preferredSize:[22,20], alignment:['right','center'] } \
				} \
			}, \
			gr2: Group { spacing: 5, alignment:['right','top'], alignChildren:['right','center'], \
				reinforcementThresholdSt: StaticText { text:'Reinforcement Threshold:' }, \
				gr21: Group { alignment:['left','top'], alignChildren:['right','center'], \
					reinforcementThresholdEt: EditText { text:'" + this.reinforcementThreshold + "', characters:5, alignment:['fill','center'], helpTip:'" + this.reinforcementThresholdHlp + "' }, \
					helpBtn: Button { text:'?', preferredSize:[22,20], alignment:['right','center'] } \
				} \
			}, \
			gr3: Group { alignment:['fill','top'], \
				aboutBtn: Button { text:'?', preferredSize:[25,20], alignment:['left','center'] }, \
				runBtn: Button { text:'Apply', preferredSize:[85,20], alignment:['right','center'] } \
			} \
		}";
		pal.gr = pal.add(res);

		// event callbacks		
		pal.gr.gr1.gr11.removalThresholdEt.onChange = function ()
		{
			if (isNaN(this.text))
			{
				this.text = removeUnnecessaryMaskKeyframes.removalThreshold;
			}
			else if (parseFloat(this.text) < 0)
			{
				this.text = 0;
			}
			removeUnnecessaryMaskKeyframes.removalThreshold = parseFloat(this.text);
		};
		
		pal.gr.gr1.gr11.helpBtn.onClick = function ()
		{
			utils.throwInfo(removeUnnecessaryMaskKeyframes.removalThresholdInfoStr, removeUnnecessaryMaskKeyframes.scriptName);
		};
		
		pal.gr.gr2.gr21.reinforcementThresholdEt.onChange = function ()
		{
			if (isNaN(this.text))
			{
				this.text = removeUnnecessaryMaskKeyframes.reinforcementThreshold;
			}
			else if (parseFloat(this.text) < 0)
			{
				this.text = 0;
			}
			removeUnnecessaryMaskKeyframes.reinforcementThreshold = parseFloat(this.text);			
		};		

		pal.gr.gr2.gr21.helpBtn.onClick = function ()
		{
			utils.throwInfo(removeUnnecessaryMaskKeyframes.reinforcementThresholdInfoStr, removeUnnecessaryMaskKeyframes.scriptName);
		};
		
		pal.gr.gr3.aboutBtn.onClick = function ()
		{
			utils.createAboutDlg(removeUnnecessaryMaskKeyframes.scriptAbout, removeUnnecessaryMaskKeyframes.scriptUsage);
		};		
		
		pal.gr.gr3.runBtn.onClick = function ()
		{
			removeUnnecessaryMaskKeyframes.filterKeys();
			app.settings.saveSetting(removeUnnecessaryMaskKeyframes.scriptName, removeUnnecessaryMaskKeyframes.removalThresholdPrefSetting, removeUnnecessaryMaskKeyframes.removalThreshold);
			app.settings.saveSetting(removeUnnecessaryMaskKeyframes.scriptName, removeUnnecessaryMaskKeyframes.reinforcementThresholdPrefSetting, removeUnnecessaryMaskKeyframes.reinforcementThreshold);				
		};

		// show user interface
		if (pal instanceof Window)
		{
			pal.center();
			pal.show();
		}
		else
		{
			pal.layout.layout(true);
		}
	};

	// Determines whether all mask keyframes have the same number of vertices
	this.validateVertices = function (maskShape)
	{
		var firstNumVerts = maskShape.keyValue(1).vertices.length;
		for (var k = 2; k <= maskShape.numKeys; k++)					
		{
			var curNumVerts = maskShape.keyValue(k).vertices.length;
			if (curNumVerts != firstNumVerts)
			{
				return false;
			}			
		}
		return true;
	};
	
	// Determines whether a vertex has moved at least X pixels ("threshold") between two keyframes
	this.isThereAMovingVertex = function (prevShape, curShape)
	{
		var foundB = false;
		
		var prevVerts = prevShape.vertices;
		var curVerts = curShape.vertices;
		var numVerts = prevVerts.length;
		
		for (var v = 0; !foundB && v < numVerts; v++)
		{
			var prevV = prevVerts[v];
			var curV = curVerts[v];
			var dist = utils.getDistance(prevV, curV);
			if (dist >= this.removalThreshold)
			{
				foundB = true;
			}
		}
		
		return foundB;
	};	
	
	// Determines the maximum distance traveled by a vertex from one shape to another
	this.getVertexMaxDistance = function (prevShape, curShape)
	{
		var maxDist = 0;
		
		var prevVerts = prevShape.vertices;
		var curVerts = curShape.vertices;
		var numVerts = prevVerts.length;
		
		for (var v = 0; v < numVerts; v++)
		{
			var prevV = prevVerts[v];
			var curV = curVerts[v];
			var dist = utils.getDistance(prevV, curV);
			if (dist > maxDist)
			{
				maxDist = dist;
			}
		}
		
		return maxDist;
	};	

	// Main function that removes keyframes during the first pass and reintroduces keyframes during the second pass
	this.filterKeys = function ()
	{
		try
		{
			// get comp
			var comp = app.project.activeItem;
			var err = this.selErr;
			if (!(comp instanceof CompItem)) throw(err);

			// get selected mask and layer
			var selProps = comp.selectedProperties;
			var mask = null;
			var layer;
			for (var i = 0; i < selProps.length; i++)
			{
				var prop = selProps[i];
				if (prop.isMask)
				{
					mask = prop;
					while (prop.parentProperty)
					{
						prop = prop.parentProperty;	
					}
					layer = prop;
					break;
				}
			}
			if (!mask) throw(err);
			
			// sanity check
			var maskShape = mask.maskShape;
			if (maskShape.numKeys < 1)
			{
				return; // nothing to do
			}
			
			// ensure the number of vertices remains constant
			var err = this.verticesErr;
			if (!this.validateVertices(maskShape)) throw(err);
									
			app.beginUndoGroup(this.scriptName);

			var err = this.processErr;
			try
			{
				// keep track of mask index to be able to reference it later on
				var maskIdx = mask.propertyIndex;
				
				// create a new mask				
				var newMaskName = mask.name + " - Filtered (" + this.removalThreshold + ", " + this.reinforcementThreshold + ")"; 
				var newMask = mask.duplicate();
				newMask.name = newMaskName;
				newMask.color = [Math.random(),Math.random(),Math.random()]; //[0.9,0.3,0.3]; // redish
				newMask.moveTo(layer.Masks.numProperties);
				newMask = layer.Masks.property(layer.Masks.numProperties); // recreate reference since moveTo() has lost it
				//newMask.selected = true;
				
				var newMaskShape = newMask.maskShape;				
				var totalKeys = newMaskShape.numKeys;
				
				// REMOVAL PASS ----------------------
				// remove "unnecessary" keys
				
				var firstPassNumRemoved = 0;
				var prevShape = newMaskShape.keyValue(1);					
				for (var k = 2; k <= newMaskShape.numKeys; k++)
				{
					var curShape = newMaskShape.keyValue(k);
					
					// if no moving vertex is found, remove the keyframe
					if (!this.isThereAMovingVertex(prevShape, curShape))
					{
						newMaskShape.removeKey(k);
						firstPassNumRemoved++;
						k--;
					}
					prevShape = curShape;
				}
				
				// REINFORCEMENT PASS ----------------
				// reintroduce keyframes if vertices move too much
				
				maskShape = layer.Masks.property(maskIdx).maskShape; // recreate reference to the original mask shape
                
				var secondPassNumAdded = 0;
				if (newMaskShape.numKeys >= 2)
				{
					var somethingToDoB = true;
					var maxIters = 100; // just to be safe in case a problem occurred
                    var iter = 0;
                    
					while (somethingToDoB && iter < maxIters)
					{
						var keyTimes = [];
						var keyValues = [];

						var prevShape = newMaskShape.keyValue(1);
						for (var k = 2; k <= newMaskShape.numKeys; k++)
						{
							// make sure keys are not too close to each other
							if (newMaskShape.keyTime(k) >= newMaskShape.keyTime(k-1) + 2*comp.frameDuration)
							{
								var curShape = newMaskShape.keyValue(k);
								var d = this.getVertexMaxDistance(prevShape, curShape);
								if (d > this.reinforcementThreshold)
								{
									var t = 0.5 * (newMaskShape.keyTime(k-1) + newMaskShape.keyTime(k));
									var nearestKeyIdx = maskShape.nearestKeyIndex(t);
									var keyT = maskShape.keyTime(nearestKeyIdx);
									var keyV = maskShape.keyValue(nearestKeyIdx);
									keyTimes.push(keyT);
									keyValues.push(keyV);
								}
							}
							prevShape = curShape;
						}
						if (!keyTimes.length)
						{
							somethingToDoB = false;
						}
						else
						{
							newMaskShape.setValuesAtTimes(keyTimes, keyValues);
							secondPassNumAdded += keyTimes.length;
						}
						iter++;
					}
				}
				
				// if there is only one keyframe remaining, remove it
				if (newMaskShape.numKeys == 1)
				{
					newMaskShape.removeKey(1);
					firstPassNumRemoved++;
				}				
				
				newMask.selected = true;
				
				// write info to Info palette
				var numRemoved = totalKeys - newMaskShape.numKeys;
				var reductionAmount = Math.round(100 * (numRemoved / totalKeys));
					
				clearOutput();
				writeLn(this.infoMsg1.replace("%d", firstPassNumRemoved));
				writeLn(this.infoMsg2.replace("%d", secondPassNumAdded));
				writeLn(this.infoMsg3.replace("%d", reductionAmount));
			}
			catch(e)
			{
				throw(err + "\n\n" + e.toString());
			}

			app.endUndoGroup();
		}
		catch(err)
		{
			utils.throwErr(err);
		}
	};

	this.run = function (thisObj)
	{
		this.buildUI(thisObj);
	};
}


// utilities
function RemoveUnnecessaryMaskKeyframesUtils()
{
	var utils = this;

	this.throwErr = function (err)
	{
		var title = File.decode($.fileName.substring($.fileName.lastIndexOf("/")+1, $.fileName.lastIndexOf(".")));
		alert(err, title, true);
	};
	
	this.throwInfo = function (err)
	{
		var title = File.decode($.fileName.substring($.fileName.lastIndexOf("/")+1, $.fileName.lastIndexOf(".")));
		alert(err, title);
	};	
	
	this.getDistance = function (ptA, ptB)
	{
		return Math.sqrt( Math.pow(ptB[0]-ptA[0],2) + Math.pow(ptB[1]-ptA[1],2) );
	};
	
	this.createAboutDlg = function (aboutStr, usageStr)
	{
		eval(unescape('%09%09%76%61%72%20%64%6C%67%20%3D%20%6E%65%77%20%57%69%6E%64%6F%77%28%22%64%69%61%6C%6F%67%22%2C%20%22%41%62%6F%75%74%22%29%3B%0A%09%20%20%20%20%20%20%09%20%20%20%20%20%20%20%09%0A%09%20%20%20%20%76%61%72%20%72%65%73%20%3D%0A%09%09%22%67%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%63%6F%6C%75%6D%6E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%61%6C%69%67%6E%43%68%69%6C%64%72%65%6E%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%5C%0A%09%09%09%70%6E%6C%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%62%65%64%70%61%6E%65%6C%27%2C%20%5C%0A%09%09%09%09%61%62%6F%75%74%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%44%65%73%63%72%69%70%74%69%6F%6E%27%2C%20%5C%0A%09%09%09%09%09%61%62%6F%75%74%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%61%62%6F%75%74%53%74%72%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%2C%20%5C%0A%09%09%09%09%75%73%61%67%65%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%55%73%61%67%65%27%2C%20%5C%0A%09%09%09%09%09%75%73%61%67%65%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%75%73%61%67%65%53%74%72%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%20%5C%0A%09%09%09%7D%2C%20%5C%0A%09%09%09%62%74%6E%73%3A%20%47%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%72%6F%77%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%62%6F%74%74%6F%6D%27%5D%2C%20%5C%0A%09%09%09%09%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%74%68%65%72%20%53%63%72%69%70%74%73%2E%2E%2E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%6C%65%66%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%2C%20%5C%0A%09%09%09%09%6F%6B%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%6B%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%72%69%67%68%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%20%5C%0A%09%09%09%7D%20%5C%0A%09%09%7D%22%3B%20%0A%09%09%64%6C%67%2E%67%72%20%3D%20%64%6C%67%2E%61%64%64%28%72%65%73%29%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%61%62%6F%75%74%53%74%72%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%3B%0A%09%09%7D%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%73%61%67%65%53%74%72%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%27%2F%67%2C%20%22%27%22%29%3B%0A%09%09%7D%3B%0A%09%09%09%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%76%61%72%20%63%6D%64%20%3D%20%22%22%3B%0A%09%09%09%76%61%72%20%75%72%6C%20%3D%20%22%68%74%74%70%3A%2F%2F%61%65%73%63%72%69%70%74%73%2E%63%6F%6D%2F%61%75%74%68%6F%72%73%2F%6D%2D%70%2F%6E%61%62%2F%22%3B%0A%09%0A%09%09%09%69%66%20%28%24%2E%6F%73%2E%69%6E%64%65%78%4F%66%28%22%57%69%6E%22%29%20%21%3D%20%2D%31%29%0A%09%09%09%7B%0A%09%20%20%20%20%20%20%20%20%09%63%6D%64%20%3D%20%22%63%6D%64%2E%65%78%65%20%2F%63%20%5C%22%73%74%61%72%74%20%22%20%2B%20%75%72%6C%20%2B%20%22%5C%22%22%3B%0A%09%09%09%7D%0A%09%09%09%65%6C%73%65%0A%09%09%09%09%63%6D%64%20%2B%3D%20%22%6F%70%65%6E%20%5C%22%22%20%2B%20%75%72%6C%20%2B%20%22%5C%22%22%3B%20%20%20%20%20%20%20%20%20%0A%09%09%09%74%72%79%0A%09%09%09%7B%0A%09%09%09%09%73%79%73%74%65%6D%2E%63%61%6C%6C%53%79%73%74%65%6D%28%63%6D%64%29%3B%0A%09%09%09%7D%0A%09%09%09%63%61%74%63%68%28%65%29%0A%09%09%09%7B%0A%09%09%09%09%61%6C%65%72%74%28%65%29%3B%0A%09%09%09%7D%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%6B%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%20%0A%09%09%7B%0A%09%09%09%64%6C%67%2E%63%6C%6F%73%65%28%29%3B%20%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%63%65%6E%74%65%72%28%29%3B%0A%09%09%64%6C%67%2E%73%68%6F%77%28%29%3B'));
	};
}


// run script
new RemoveUnnecessaryMaskKeyframes().run(this);
